; Incremental mode for ISPELL uses running in background ispell process
; to check current buffer during pauses between text modifications.
; It underlines words which are not recognized by ispell as correct ones,
; so it need EMACS started under window system to make underline possible. 
; 
; After loading EMACS menu Edit/Spell must contain item showing
; current state of this mode, same time used to switch it ON/OFF.
; 
; This mode automatically spell-check any active buffer.
; When it is not desirable You can redefine  incr-buffer-p  function
; to spell-check only selected buffers. 
; 
; This mode must automatically switched off during normal interactive
; ispell sessions. Switching this mode off removes all underlines.
; 
; 'incr-ispell' was tested with EMACS 19-31,34.


; Please compile this file (for speed) before using: type
;     emacs -batch -q -no-site-file -f batch-byte-compile incr-ispell.el
; and put obtained incr-ispell.elc into some proper emacs-lisp directory

; Add into Your .emacs file following line:  
;     (load "incr-ispell")

; If emacs don't want load file, check it's load-path. Add line like
;     (setq load-path (cons "your_elisp_dir" load-path))
; into Your .emacs file to make directory with incr-ispell.elc loadable.

; Attention ! This mode is partially based on ispell.el specific, 
;             so it possibly will not work with modified one  !!!

; History:

; 05-oct-97:  Protect added again 'kill-all-local-variables' which 
;             lead underlined overlays remains after switching into
;             another mode.
;
;	      Calling 'ispell-accept-buffer-local-defs' 
;             in 'incr-ispell-start' to update state and make function 
;             'ispell-batch-word-ported' able to use current dictionary
;             setting.
;             This bug lead to failure in automatically starting ispell 
;             in some cases

; 23-sep-97:  Reentrant re-spelling denied (why emacs timer do it?)
;             Check is window-system present added. 
;             incr-buffer-p  predicate added.

; 22-sep-97:  First lazy functional highlighting algorithm version 
;             where run-with-idle-timer function used
;             and two-level memorization algorithm was implemented:
;             for lines and for words. 

;             For AMD/133 PC re-spelling after local changes takes about
;             0.01 sec CPU time; after global changes -- about 0.1 sec.
;             However, highlighting of text with large number of bad words
;             or text with non-cached words take much more time 

; 15-sep-97:  Zero version, when only newly typed/changed words stressed


;; Require used packages
(require 'cl)
(require 'ispell)
(require 'timer) 
(eval-when-compile (load "cl-extra"))



(defvar incr-idle-timeout 2)  ; secs between last emacs operation and remap

(defvar incr-screens-up   1)

(defvar incr-screens-down 1)  ; excursion up/down to obtain region to remap

(defvar incr-word-cache-size 10000) ; max words in internal cache hash table

(defun incr-buffer-p (name)
; Use body like this to check only some buffers
; (string-match "\\.html\\|\\.txt\\|\\.doc" name)
  t) 


(defun batch-ispell-word-ported (&optional following quietly continue)
  "Ported version of ISPELL-WORD (see ispell.el)"
  (interactive (list nil nil current-prefix-arg))
  (if continue
      (ispell-continue)
    (if (interactive-p)
	(setq following ispell-following-word
	      quietly ispell-quietly))
    (ispell-accept-buffer-local-defs)	; use the correct dictionary
    (let ((cursor-location (point))	; retain cursor location
	  (word (ispell-get-word following))
	  start end poss replace)
      ;; destructure return word info list.
      (setq start (car (cdr word))
	    end (car (cdr (cdr word)))
	    word (car word))

      ;; now check spelling of word.
      (or quietly
	  (message "Checking spelling of %s..."
		   (funcall ispell-format-word word)))
      (process-send-string ispell-process "%\n") ;put in verbose mode
      (process-send-string ispell-process (concat "^" word "\n"))
      ;; wait until ispell has processed word
      (while (progn
	       (accept-process-output ispell-process)
	       (not (string= "" (car ispell-filter)))))
      ;;(process-send-string ispell-process "!\n") ;back to terse mode.
      (setq ispell-filter (cdr ispell-filter))
      (if (listp ispell-filter)
	  (setq poss (ispell-parse-output (car ispell-filter))))
      (cond ((eq poss t)
	     (or quietly
		 (message "%s is correct" (funcall ispell-format-word word))))
	    ((stringp poss)
	     (or quietly
		 (message "%s is correct because of root %s"
			  (funcall ispell-format-word word)
			  (funcall ispell-format-word poss))))
	    ((null poss) (message "Error in ispell process"))
	    (ispell-check-only		; called from ispell minor mode.
	     (beep))
	    (t nil)))))


(defun batch-ispell-word (pos)
  "Spell-check word near position pos"
  (let* ((save (point))
	 (dummy (goto-char pos))
	 (ret (batch-ispell-word-ported nil t)))
    (goto-char save)
    ret))


(defun incr-underline-overlay (start end)
  "Create region, underline it and return it as a result"
  (let ((over (make-overlay start end)))
    (overlay-put over 'face 'underline)
    over))


;; Cache/Hash table for spelled words
(defvar *incr-word-cache*)

(defun incr-reset-word-cache ()
  "Reinitialize word cache"
  (let ((word-0 `(nil . nil)))
    (setq *incr-word-cache* `(,(make-hash-table :test 'equal) 
			      0                ; word count
			      ,word-0          ; first word
			      ,word-0))))      ; last  word 

(defun incr-add-word (hash count fst lst word info)
  "Add unknown word into word cache"
  (if (>= count incr-word-cache-size)
      (progn
	(remhash (car fst) hash)
	(setq fst (cdr fst))
	(decf count)))
  (cl-puthash word info hash)
  (setq lst (setcdr lst `(,word)))
  (incf count)
  `(,hash ,count ,fst ,lst))


(defun incr-get-info (word pos)
  "Get information about word in position pos"
  (let ((info (cl-gethash word (first *incr-word-cache*) 'none)))
    (if (not (eq 'none info)) info
      (setq *incr-word-cache* 
	    (apply #'incr-add-word 
		   `(,@*incr-word-cache* ,word ,(incr-gen-info word pos))))
      (incr-get-info word pos))))


(defun incr-gen-info (word pos)
  "Generate information about word"
  (batch-ispell-word pos))

(mapcar #'(lambda (sym) 
	    (make-variable-buffer-local sym)
	    (put sym 'permanent-local t))
	    '(incr-overlays               ; Overlays for 'bad' words
	      incr-old-lines))            ; Saved cache of lines

(defun incr-remap-all ()
  "Main function doing incremental spelling"
   
  ; Do initialization of buffer data for XEMACS
  (if (not (boundp 'incr-overlays))
      (setq incr-overlays nil  incr-old-lines  nil))
  
  ; Remove all our overlays
  (mapcar #'delete-overlay incr-overlays)
  (setq incr-overlays nil)
  
  (save-excursion
    (let* ((save (point))
	   (c (* incr-screens-up (window-height)))
	   (new  nil)
	   (mrk  nil))
    
   ;; (setq tt (current-time-decs))
    
      (while (and (> (point) (point-min)) (> (decf c) 0))
	(re-search-backward "\n" (point-min) 'move))

      (setq c (* (+ incr-screens-up 1 incr-screens-down) (window-height)))
      (beginning-of-line)
      (while (and (< (point) (point-max)) (> (decf c) 0))
	(setq mrk (set-marker (make-marker) (point)))
	(re-search-forward "\n" (point-max) 'move)
	(push (list mrk (buffer-substring (marker-position mrk) (point))) 
	      new))

      (mapcar #'incr-str-map
	      (setq incr-old-lines 
		    (incr-m-slice incr-old-lines (reverse new))))
      
      (goto-char save)
   
   ;; (message (format "%s" (- (current-time-decs) tt)))
   
      )))
    

; Flag we already doing incr-remap-all
(defvar incr-remap-in-progress nil)

(defun incr-idle-handler ()
  "Protect against reenterant calls and non-our buffers"
  (if (and (incr-buffer-p (buffer-name (current-buffer)))
	   (not incr-remap-in-progress))
      (progn
	(setq incr-remap-in-progress t)
	(unwind-protect 
	    (incr-remap-all)
	  (setq incr-remap-in-progress nil)))))



(defun incr-str-map (stri) 
  "Map string-information overlays"
  (mapcar #'(lambda (over) 
	      (push (apply #'incr-underline-overlay 
			   (mapcar #'(lambda (x) 
				       (+ x (marker-position (first stri))))
				   over))
		    incr-overlays))
	  (third stri)))


(defun incr-m-slice (old new)
  "Slice two assoc lists of old (marker string info) 
                        and new (marker string)"
  (let ((res nil))
    (while new
      (cond ((and old 
		  (< (caar old) (caar new)))
	     (setq old (cdr old)))
	    ((and old 
		  (= (caar old) (caar new)) 
		  (equal (cadar old) (cadar new)))
	     (push (car old) res)
	     (setq new (cdr new)))
	    (t 
	     (push `(,@(car new) ,(incr-str-info (caar new) (cadar new))) res)
	     (setq new (cdr new)))))
    (reverse res)))
	   

(defun incr-str-info (mrk str) 
  "Generate information about string; i.e. list of overlays"
  (let ((end (+ mrk (length str)))
	(finish nil)
	(over-list nil)
	(prev-pos nil))
    (goto-char mrk)
    
    (flet ((error (&rest x) (setq finish t)))
      (while (not 
	      (setq finish (or finish 
			       (>= (point) end))))
	(let ((winfo 
	       (ispell-get-word t)))
	  (setq finish (or finish 
			   (eq prev-pos (second winfo))
			   (> (point) end)))
	  (setq prev-pos (second winfo))
	  (if (and (not finish) 
		   (not (incr-get-info (first winfo) (second winfo)))
		   (not finish)) ; twice because some error may be 
	      (push (mapcar #'(lambda (x) (- x mrk)) (cdr winfo)) 
		    over-list)))))
    over-list))
		

(defvar incr-ispell-is-on)

(defun incr-ispell-start ()
  "Make incr-remap-all calling sometimes"
  (ispell-accept-buffer-local-defs) ;; Update real state from local state
  (if (not incr-ispell-is-on)
      (progn
	(run-with-idle-timer incr-idle-timeout t #'incr-idle-handler)
	(setq incr-ispell-is-on t)

	(define-key ispell-menu-map [ispell-incr-ispell]
	  '("Incremental mode now is  ON" 
	    . 
	    (lambda () 
	      (interactive) 
	      (incr-ispell-stop)
	      (message "Incremental ISPELL mode is now OFF")))))))


(defun incr-ispell-stop ()
  "Stop incremental spelling"
  (cancel-function-timers #'incr-idle-handler)
  (setq incr-remap-in-progress nil)
  
  ;; Clear all buffer-specific data
  (let ((save (current-buffer)))
    (mapcar #'(lambda (buf)
		(set-buffer buf)
		(if (boundp 'incr-overlays)
		    (mapcar #'delete-overlay incr-overlays))
		(setq incr-overlays nil  incr-old-lines nil))
	    (buffer-list))
    (set-buffer save))
  
  ;; Reset word cache (one for all buffers) and 
  (incr-reset-word-cache)
  (setq incr-ispell-is-on nil)
  
  (define-key ispell-menu-map [ispell-incr-ispell]
    '("Incremental mode now is  OFF" 
      . 
      (lambda () 
	(interactive) 
	(incr-ispell-start)
	(message "Incremental ISPELL mode is now ON")))))
  


(if (not window-system)
    (message "incr-ispell: can't work without window system") 
  (progn

    ; Make initial state OFF
    (incr-ispell-stop)

    ; Redefine interactive ispell functions

    (defvar  saved-ispell-region nil)
    (if (not saved-ispell-region)
	(progn 
	  (setq saved-ispell-region t)
	  (setf (symbol-function 'saved-ispell-region) 
		(symbol-function 'ispell-region))
	  (setf (symbol-function 'ispell-region)
		#'(lambda (reg-start reg-end)
		    "Interactively check a region for spelling errors."
		    (interactive "r")	; Don't flag errors on read-only bufs.
		    (let ((state incr-ispell-is-on))
		      (if state (incr-ispell-stop))
		      (unwind-protect
			  (saved-ispell-region reg-start reg-end)
			(if state (incr-ispell-start))))))))


    (defvar  saved-ispell-word nil)
    (if (not saved-ispell-word)
	(progn 
	  (setq saved-ispell-word t)
	  (setf (symbol-function 'saved-ispell-word) 
		(symbol-function 'ispell-word))
	  (setf (symbol-function 'ispell-word)
		#'(lambda (&optional following quietly continue) 
		    (interactive (list nil nil current-prefix-arg))
		    (let ((state incr-ispell-is-on))
		      (if state (incr-ispell-stop))
		      (unwind-protect
			  (saved-ispell-word following quietly continue)
			(if state (incr-ispell-start))))))))
    

    (defvar  saved-ispell-change-dictionary nil)
    (if (not saved-ispell-change-dictionary)
	(progn 
	  (setq saved-ispell-change-dictionary t)
	  (setf (symbol-function 'saved-ispell-change-dictionary) 
		(symbol-function 'ispell-change-dictionary))
	  (setf (symbol-function 'ispell-change-dictionary)
		#'(lambda (dict &optional arg) 
		    (interactive
		     (list (completing-read
		  "Use new dictionary (RET for current, SPC to complete): "
		  (cons (cons "default" nil) ispell-dictionary-alist) nil t)
			   current-prefix-arg))
		    (let ((state incr-ispell-is-on))
		      (if state (incr-ispell-stop))
		      (unwind-protect
			  (saved-ispell-change-dictionary dict arg)
			(if state (incr-ispell-start))))))))))

