/*
 * Copyright: GNU Public License 2 applies
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * CDDA2WAV (C) Heiko Eissfeldt heiko@colossus.escape.de
 * CDDB routines (C) Ti Kan and Steve Scherf
 */
#include "config.h"
#include <standard.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined (HAVE_UNISTD_H) && (HAVE_UNISTD_H == 1)
#include <sys/types.h>
#include <unistd.h>		/* sleep */
#endif
#include <ctype.h>
#include <errno.h>

#define CD_EXTRA
#undef DEBUG_XTRA
#ifdef CD_EXTRA
#include <stdlib.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#endif

#include "byteorder.h"
#include "mycdrom.h"
#include "interface.h"
#include "cdda2wav.h"
#include "global.h"
#include "toc.h"
#include "ringbuff.h"

int have_CD_text;
int have_CD_extra;

static void UpdateTrackData	__PR((int p_num));
static void UpdateIndexData	__PR((int p_num));
static void UpdateTimeData	__PR((int p_min, int p_sec, int p_frm));
static unsigned int is_cd_extra	__PR((unsigned int no_tracks));
static unsigned int get_end_of_last_audio_track	__PR((unsigned int mult_off, unsigned int no_tracks));
static int cddb_sum		__PR((int n));
static void emit_cddb_form	__PR((char *fname_baseval));
static void dump_cdtext_info	__PR((void));
static void dump_extra_info	__PR((void));
static int GetIndexOfSector	__PR((unsigned int sec));
static int binary_search	__PR((int searchInd, unsigned int Start, unsigned int End));

static unsigned char g_track=0xff, g_index=0xff;	/* current track, index */
static unsigned char g_minute=0xff, g_seconds=0xff;	/* curr. minute, second */
static unsigned char g_frame=0xff;			/* current frame */


/* print the track currently read */
static void UpdateTrackData (p_num)
	unsigned char p_num;
{
  if (global.quiet == 0) { 
    fprintf (stderr, "\ntrack: %.2d, ", p_num); fflush(stderr);
  }
  g_track = p_num;
}


/* print the index currently read */
static void UpdateIndexData (p_num)
	unsigned char p_num;
{
  if (global.quiet == 0) { 
    fprintf (stderr, "index: %.2d\n", p_num); fflush(stderr);
  }
  g_index = p_num;
}


/* print the time of track currently read */
static void UpdateTimeData (p_min, p_sec, p_frm)
	unsigned char p_min;
	unsigned char p_sec;
	unsigned char p_frm;
{
  if (global.quiet == 0) {
    fprintf (stderr, "time: %.2d:%.2d.%.2d\r", p_min, p_sec, p_frm); 
    fflush(stderr);
  }
  g_minute = p_min;
  g_seconds = p_sec;
  g_frame = p_frm;
}

void AnalyzeQchannel ( frame )
	unsigned frame;
{
    subq_chnl *sub_ch;

    if (trackindex_disp != 0) {
	sub_ch = ReadSubQ( GET_POSITIONDATA,0);
	/* analyze sub Q-channel data */
	if (sub_ch->track != g_track ||
	    sub_ch->index != g_index) {
	    UpdateTrackData (sub_ch->track);
	    UpdateIndexData (sub_ch->index);
	}
    }
    frame += 150;
    UpdateTimeData ((unsigned char) (frame / (60*75)), 
		    (unsigned char) ((frame % (60*75)) / 75), 
		    (unsigned char) (frame % 75));
}

unsigned cdtracks = 0;

long GetStartSector ( p_track )
	unsigned long p_track;
{
  unsigned long i;

  for (i = 0; i < cdtracks; i++) {
    if (g_toc [i].bTrack == p_track) {
      unsigned long dw = g_toc [i].dwStartSector;
      if ((g_toc [i].bFlags & CDROM_DATA_TRACK) != 0)
	return -1;
      return dw;
    }
  }

  return -1;
}


long GetEndSector ( p_track )
	unsigned long p_track;
{
  unsigned long i;

  for ( i = 1; i <= cdtracks; i++ ) {
    if ( g_toc [i-1].bTrack == p_track ) {
      unsigned long dw = g_toc [i].dwStartSector;
      return dw-1;
    }
  }

  return -1;
}

long FirstTrack ( )
{
  unsigned long i;
  
  for ( i = 0; i < cdtracks; i++ ) {
    if ( g_toc [i].bTrack != CDROM_LEADOUT &&
	( g_toc [i].bFlags & CDROM_DATA_TRACK ) == 0 )
      return g_toc [i].bTrack;
  }
  return 0;
}

long GetLastSectorOnCd( p_track )
	unsigned long p_track;
{
  unsigned long i;
  long LastSec = 0;

  for ( i = 1; i <= cdtracks; i++ ) {
    if ( g_toc [i-1].bTrack < p_track )
      continue;

    /* break if a nonaudio track follows */
    if ( (g_toc [i-1].bFlags & CDROM_DATA_TRACK) != 0) break;
    LastSec = GetEndSector ( g_toc [i-1].bTrack ) + 1;
  }
  return LastSec;
}

int GetTrack( sector )
	unsigned long sector;
{
  unsigned long i;
  for (i = 0; i < cdtracks; i++) {
    if (g_toc[i  ].dwStartSector <= sector &&
	g_toc[i+1].dwStartSector > sector)
      return (g_toc [i].bFlags & CDROM_DATA_TRACK) != 0 ? -1 : (int)(i+1);
  }
  return -1;
}

int CheckTrackrange( from, upto )
	unsigned long from;
	unsigned long upto;
{
  unsigned long i;

  for ( i = 1; i <= cdtracks; i++ ) {
    if ( g_toc [i-1].bTrack < from )
      continue;

    if ( g_toc [i-1].bTrack == upto )
      return 1;

    /* break if a nonaudio track follows */
    if ( (g_toc [i-1].bFlags & CDROM_DATA_TRACK) != 0)
      return 0;
  }
  /* track not found */
  return 0;
}

unsigned
find_an_off_sector __PR((unsigned lSector, unsigned SectorBurstVal));

unsigned find_an_off_sector(lSector, SectorBurstVal)
	unsigned lSector;
	unsigned SectorBurstVal;
{
	long track_of_start = GetTrack(lSector);
	long track_of_end = GetTrack(lSector + SectorBurstVal -1);
	long start = GetStartSector(track_of_start);
	long end = GetEndSector(track_of_end);

	if (lSector - start > end - lSector + SectorBurstVal -1)
		return start;
	else
		return end;
}

#ifdef CD_EXTRA
/**************** CD-Extra special treatment *********************************/

static unsigned char Extra_buffer[CD_FRAMESIZE_RAW];

#if defined CDROMMULTISESSION
static int tmp_fd;
#endif

static int Read_CD_Extra_Info __PR(( unsigned long sector));
static int Read_CD_Extra_Info(sector)
	unsigned long sector;
{
#ifdef DEBUG_XTRA
  fprintf(stderr, "debug: Read_CD_Extra_Info at sector %lu\n", sector);
#endif
  ReadCdRomData( Extra_buffer, sector, 1);

  /* If we are unlucky the drive cannot handle XA sectors by default.
     We try to compensate by ignoring the first eight bytes.
     Of course then we lack the last 8 bytes of the sector...
   */

#ifdef DEBUG_XTRA
  { int ijk;
    for (ijk = 0; ijk < 96; ijk++) {
      fprintf(stderr, "%02x  ", Extra_buffer[ijk]);
      if ((ijk+1) % 16 == 0) fprintf(stderr, "\n");
    }
  } 
#endif

  if (Extra_buffer[0] == 0)
    movebytes(Extra_buffer +8, Extra_buffer, CD_FRAMESIZE - 8);
  return 0;
}

static void Read_Subinfo __PR(( unsigned pos, unsigned length));
static void Read_Subinfo(pos, length)
	unsigned pos;
	unsigned length;
{
  unsigned num_infos, num;
  unsigned char *Subp, *orgSubp;
  unsigned this_track = 0xff;
#ifdef DEBUG_XTRA
  unsigned char *up;
  unsigned char *sp;
  unsigned u;
  unsigned short s;
#endif

  length += 8;
  length = (length + CD_FRAMESIZE_RAW-1) / CD_FRAMESIZE_RAW;
  length *= CD_FRAMESIZE_RAW;
  orgSubp = Subp = (unsigned char *) malloc(length);

  if (Subp == NULL) {
    fprintf(stderr, "Read_Subinfo alloc error(%d)\n",length);
    goto errorout;
  }

  ReadCdRomData( Subp, pos, 1);

  num_infos = Subp[45]+(Subp[44] << 8);
#ifdef DEBUG_XTRA
  fprintf(stderr, "subinfo version %c%c.%c%c, %d info packets\n",
	  Subp[8],
	  Subp[9],
	  Subp[10],
	  Subp[11],
	  num_infos);
#endif
  length -= 46;
  Subp += 46;
  for (num = 0; num < num_infos && length > 0; num++) {
    static const char *infopacketID[] = { "0", 
			      "track identifier", 
			      "album title",
			      "universal product code", 
			      "international standard book number",
			      "copyright",
			      "track title",
			      "notes",
			      "main interpret",
			      "secondary interpret",
			      "composer",
			      "original composer",
			      "creation date",
			      "release  date",
			      "publisher",
			      "0f",
			      "isrc audio track",
			      "isrc lyrics",
			      "isrc pictures",
			      "isrc MIDI data",
			      "14", "15", "16", "17", "18", "19",
			      "copyright state SUB_INFO",
			      "copyright state intro lyrics",
			      "copyright state lyrics",
			      "copyright state MIDI data",
			      "1e", "1f",
			      "intro lyrics",
			      "pointer to lyrics text file and length", 
			      "22", "23", "24", "25", "26", "27", "28",
			      "29", "2a", "2b", "2c", "2d", "2e", "2f",
			      "still picture descriptor",
			      "31",
			      "32", "33", "34", "35", "36", "37", "38",
			      "39", "3a", "3b", "3c", "3d", "3e", "3f",
			      "MIDI file descriptor",
			      "genre code",
			      "tempo",
			      "key"
			     };
    unsigned id = *Subp;
    unsigned len = *(Subp +1);

    if (id >= sizeof(infopacketID)/sizeof(const char *)) {
      fprintf(stderr, "Off=%4d, ind=%2d/%2d, unknown Id=%2u, len=%2u ",
		Subp - orgSubp, num, num_infos, id, len);
      Subp += 2 + 1;
      length -= 2 + 1;
      break;
    }

    switch (id) {
    case 1:    /* track nummer or 0 */
      this_track = 10 * (*(Subp + 2) - '0') + (*(Subp + 3) - '0');
      break;

    case 0x02: /* album title */
	if (global.disctitle == NULL) {
	    global.disctitle = (unsigned char *) malloc(len + 1);
	    if (global.disctitle != NULL) {
               memcpy(global.disctitle, Subp + 2, len);
	       global.disctitle[len] = '\0';
            }
        }
      break;
    case 0x06: /* track title */
	if (this_track > 0 && this_track < 100
	    && global.tracktitle[this_track-1] == NULL) {
            global.tracktitle[this_track-1] = (unsigned char *) malloc(len + 1);
            if (global.tracktitle[this_track-1] != NULL) {
               memcpy(global.tracktitle[this_track-1], Subp + 2, len);
               global.tracktitle[this_track-1][len] = '\0';
            }
        }
      break;
    case 0x05: /* copyright message */
	if (global.copyright_message == NULL) {
	    global.copyright_message = (unsigned char *) malloc(len + 1);
	    if (global.copyright_message != NULL) {
               memcpy(global.copyright_message, Subp + 2, len);
	       global.copyright_message[len] = '\0';
            }
        }
      break;
    case 0x08: /* creator */
	if (global.creator == NULL) {
	    global.creator = (unsigned char *) malloc(len + 1);
	    if (global.creator != NULL) {
               memcpy(global.creator, Subp + 2, len);
	       global.creator[len] = '\0';
            }
        }
      break;
#if 0
    case 0x03:
    case 0x04:
    case 0x07:
    case 0x09:
    case 0x0a:
    case 0x0b:
    case 0x0c:
    case 0x0d:
    case 0x0e:
    case 0x0f:
      fprintf(stderr, "%s: %*.*s\n",infopacketID[id], (int) len, (int) len, (Subp +2));
      break;
#ifdef DEBUG_XTRA
    case 0x1a:
    case 0x1b:
    case 0x1c:
    case 0x1d:
	fprintf(stderr, "%s %scopyrighted\n", infopacketID[id], *(Subp + 2) == 0 ? "not " : "");
      break;

    case 0x21:
      fprintf(stderr, "lyrics file beginning at sector %u",
	      (unsigned) GET_BE_UINT_FROM_CHARP(Subp + 2));
      if (len == 8)
	fprintf(stderr, ", having length: %u\n", 
                (unsigned) GET_BE_UINT_FROM_CHARP(Subp + 6));
      else
	fputs("\n", stderr);
      break;

    case 0x30:
      sp = Subp + 2;
      while (sp < Subp + 2 + len) {
      /*while (len >= 10) {*/
        s = be16_to_cpu((*(sp)) | (*(sp) << 8));
        fprintf(stderr, "%04x, ", s);
	sp += 2;
        up = sp;
	switch (s) {
	case 0:
	break;
	case 4:
	break;
	case 5:
	break;
	case 6:
	break;
        }
        u = GET_BE_UINT_FROM_CHARP(up);
        fprintf(stderr, "%04lx, ", (long) u);
        up += 4;
        u = GET_BE_UINT_FROM_CHARP(up);
        fprintf(stderr, "%04lx, ", (long) u);
        up += 4;
	sp += 8;
      }
      fputs("\n", stderr);
      break;

    case 0x40:
      fprintf(stderr, "MIDI file beginning at sector %u",
	      (unsigned) GET_BE_UINT_FROM_CHARP(Subp + 2));
      if (len == 8)
	fprintf(stderr, ", having length: %u\n", 
		(unsigned) GET_BE_UINT_FROM_CHARP(Subp + 6));
      else
	fputs("\n", stderr);
      break;

    case 0x42:
      fprintf(stderr, "%s: %d beats per minute\n",infopacketID[id], *(Subp + 2));
      break;
    case 0x41:
      if (len == 8)
        fprintf(stderr, "%s: %x, %x, %x, %x, %x, %x, %x, %x\n",
		infopacketID[id],
		*(Subp + 2),
		*(Subp + 3),
		*(Subp + 4),
		*(Subp + 5),
		*(Subp + 6),
		*(Subp + 7),
		*(Subp + 8),
		*(Subp + 9)
	);
      else
        fprintf(stderr, "%s:\n",infopacketID[id]);
      break;
    case 0x43:
      fprintf(stderr, "%s: %x\n",infopacketID[id], *(Subp + 2));
      break;
    default:
      fprintf(stderr, "%s: %*.*s\n",infopacketID[id], (int) len, (int) len, (Subp +2));
#endif
#endif
    }

    if (len & 1) len++;
    Subp += 2 + len;
    length -= 2 + len;
  }

/* cleanup */

  free(orgSubp);

  return;

errorout:
  exit(2);
}

static unsigned session_start;
#endif

/*
   A Cd-Extra is detected, if it is a multisession CD with
   only audio tracks in the first session and a data track
   in the last session.
 */
static unsigned is_cd_extra(no_tracks)
	unsigned no_tracks;
{
  unsigned mult_off;
#if defined CDROMMULTISESSION
  /*
   * FIXME: we would have to do a ioctl (CDROMMULTISESSION)
   *        for the cdrom device associated with the generic device
   *	    not just AUX_DEVICE
   */
  struct cdrom_multisession ms_str;

  if (interface == GENERIC_SCSI)
    tmp_fd = open (global.aux_name, O_RDONLY);
  else
    tmp_fd = global.cooked_fd;

  if (tmp_fd != -1) {
    int result;

    ms_str.addr_format = CDROM_LBA;
    result = ioctl(tmp_fd, CDROMMULTISESSION, &ms_str);
    if (result == -1) {
      if (global.verbose != 0)
        perror("multi session ioctl not supported: ");
    } else {
#ifdef DEBUG_XTRA
  fprintf(stderr, "current ioctl multisession_offset = %u\n", ms_str.addr.lba);
#endif
	if (ms_str.addr.lba > 0)
	  return ms_str.addr.lba;
    }
  }
#endif
  mult_off = (no_tracks > 2 && !IS_AUDIO(no_tracks-2) && IS_AUDIO(no_tracks-3))
                      ? g_toc[no_tracks-2].dwStartSector : 0;
#ifdef DEBUG_XTRA
  fprintf(stderr, "current guessed multisession_offset = %u\n", mult_off);
#endif
  return mult_off;
}

#define SESSIONSECTORS (152*75)
/*
   The solution is to read the Table of Contents of the first
   session only (if the drive permits that) and directly use
   the start of the leadout. If this is not supported, we subtract
   a constant of SESSIONSECTORS sectors (found heuristically).
 */
static unsigned get_end_of_last_audio_track(mult_off, no_tracks)
	unsigned mult_off;
	unsigned no_tracks;
{
   unsigned retval;

   /* Try to read the first session table of contents.
      This works for Sony and mmc type drives. */
   if (ReadLastAudio && (retval = ReadLastAudio( no_tracks)) != 0) {
     return retval;
   } else {
     return mult_off - SESSIONSECTORS;
   }
}

/* The Table of Contents needs to be corrected if we
   have a CD-Extra. In this case all audio tracks are
   followed by a data track (in the second session).
   Unlike for single session CDs the end of the last audio
   track cannot be set to the start of the following
   track, since the lead-out and lead-in would then
   errenously be part of the audio track. This would
   lead to read errors when trying to read into the
   lead-out area.
   So the length of the last track in case of Cd-Extra
   has to be fixed.
 */
int FixupTOC(no_tracks)
	unsigned no_tracks;
{
#ifdef CD_EXTRA
    unsigned mult_off;

    /* get the multisession offset in sectors */
    mult_off = is_cd_extra(no_tracks);

#ifdef DEBUG_XTRA
    fprintf(stderr, "current multisession_offset = %u\n", mult_off);
#endif
    if (mult_off > 100) { /* the offset has to have a minimum size */
      int j;
      unsigned real_end;

      /* believe the multisession offset :-) */
      /* adjust end of last audio track to be in the first session */
      real_end = get_end_of_last_audio_track(mult_off, no_tracks);
#ifdef DEBUG_XTRA
    fprintf(stderr, "current end = %u\n", real_end);
#endif
      for (j = no_tracks-2; j > 0; j--) {
	if (!IS_AUDIO(j) && IS_AUDIO(j-1)) {
	  if (g_toc[j].dwStartSector > real_end) {
	    session_start = mult_off;
#ifdef DEBUG_XTRA
    fprintf(stderr, "reading info sector %u (+75)\n", g_toc[j].dwStartSector + 75);
#endif
	    Read_CD_Extra_Info(g_toc[j].dwStartSector + 75);
            /* set start of track to beginning of lead-out */
#ifdef DEBUG_XTRA
    fprintf(stderr, "setting end of session to %u\n", real_end);
#endif
	    g_toc[j].dwStartSector = real_end;
	    return 1;
	  }
	  break;
	}
      }
    }
#endif
    return 0;
}

static int cddb_sum(n)
	int n;
{
  int ret;

  for (ret = 0; n > 0; n /= 10) {
    ret += (n % 10);
  }

  return ret;
}

UINT4 calc_cddb_id()
{
  UINT4 i;
  UINT4 t = 0;
  UINT4 n = 0;

  for (i = 0; i < cdtracks; i++) {
    n += cddb_sum(g_toc[i].dwStartSector/75 + 2);
  }

  t = g_toc[i].dwStartSector/75 - g_toc[0].dwStartSector/75;

  return (n % 0xff) << 24 | (t << 8) | cdtracks;
}

#if defined CDDB_SUPPORT

static void emit_cddb_form(fname_baseval)
	char *fname_baseval;
{
  unsigned i;
  unsigned first_audio;
  FILE *cddb_form;
  char fname[200];
  char *pp;

  if (fname_baseval == NULL || fname_baseval[0] == 0)
	return;

  strncpy(fname, fname_baseval, sizeof(fname) -1);
  fname[sizeof(fname) -1] = 0;
  pp = strrchr(fname, '.');
  if (pp == NULL) {
    pp = fname + strlen(fname);
  }
  strncpy(pp, ".cddb", sizeof(fname) - 1 - (pp - fname));

  cddb_form = fopen(fname, "w");
  if (cddb_form == NULL) return;

  first_audio = FirstTrack();
  fprintf( cddb_form, "# xmcd\n#\n");
  fprintf( cddb_form, "# Track frame offsets:\n#\n");
  for (i = first_audio - 1; i < cdtracks; i++) {
    if (!IS_AUDIO(i)) break;
    fprintf( cddb_form, "# %u\n", 150 + g_toc[i].dwStartSector);
  }
  fprintf( cddb_form, "#\n# Disc length: %lu seconds\n#\n", 
           (GetLastSectorOnCd(first_audio) - GetStartSector(first_audio)) / 75);
  fprintf( cddb_form, "# Revision: 0\n" );
  fprintf( cddb_form, "# Submitted via: cdda2wav ");
  fprintf( cddb_form, VERSION);
  fprintf( cddb_form, "\n" );

  fprintf( cddb_form, "DISCID=%8lx\n", global.cddb_id);
  if (global.disctitle == NULL && global.creator == NULL) {
    fprintf( cddb_form, "DTITLE=\n");
  } else {
    if (global.creator == NULL) {
      fprintf( cddb_form, "DTITLE=%s\n", global.disctitle);
    } else if (global.disctitle == NULL) {
      fprintf( cddb_form, "DTITLE=%s\n", global.creator);
    } else {
      fprintf( cddb_form, "DTITLE=%s / %s\n", global.creator, global.disctitle);
    }
  }

  for (i = first_audio - 1; i < cdtracks; i++) {
    if (!IS_AUDIO(i)) break;
    if (global.tracktitle[i] != NULL) {
      fprintf( cddb_form, "TTITLE%d=%s\n", i, global.tracktitle[i]);
    } else {
      fprintf( cddb_form, "TTITLE%d=\n", i);
    }
  }

  if (global.copyright_message == NULL) {
    fprintf( cddb_form, "EXTD=\n");
  } else {
    fprintf( cddb_form, "EXTD=Copyright %s\n", global.copyright_message);
  }

  for (i = first_audio - 1; i < cdtracks; i++) {
    if (!IS_AUDIO(i)) break;
    fprintf( cddb_form, "EXTT%d=\n", i);
  }
  fprintf( cddb_form, "PLAYORDER=\n");
  fclose( cddb_form );
}

#ifdef NOTYET
/* network access for cddb servers */
static int get_list_of_servers()
{
/*
CDDB_SITES_SERVER=cddb.cddb.com
CDDB_SITES_SERVERPORT=8880
*/
return 0;
}

static int scan_servers(cddb_id)
	unsigned cddb_id;
{
return 0;
}

int resolve_id(cddb_id)
	unsigned cddb_idr;)
{
  unsigned servers;

  servers = get_list_of_servers();

  if (servers == 0) return 0;

  return scan_servers(cddb_id);
}
#endif
#endif

static void dump_cdtext_info()
{
}

static void dump_extra_info()
{
#ifdef CD_EXTRA
  unsigned char *p;
  unsigned pos, length;

  p = Extra_buffer + 48;
  while (*p != 0) {
    pos    = GET_BE_UINT_FROM_CHARP(p+2);
    length = GET_BE_UINT_FROM_CHARP(p+6);
    pos += session_start;

    if (global.verbose != 0) {
	fprintf(stderr, "Language: %c%c (as defined by ISO 639)", *p, *(p+1));
#ifdef DEBUG_XTRA
	fprintf(stderr, " at sector %u, len=%u (sessionstart=%u)", pos, length, session_start);
#endif
	fputs("\n", stderr);
    }
    /* dump this entry */
    Read_Subinfo(pos, length);
    p += 10;

    if (p + 9 > (Extra_buffer + CD_FRAMESIZE))
      break;
  }
#endif
}

void DisplayToc ( )
{
  unsigned i;
  unsigned long dw;
  unsigned mins;
  unsigned secnds;
  unsigned centi_secnds;	/* hundreds of a second */
  int count_audio_trks;


  /* get total time */
  dw = (unsigned long) g_toc[cdtracks].dwStartSector + 150;
  mins	       =       dw / ( 60*75 );
  secnds       =     ( dw % ( 60*75 ) ) / 75;
  centi_secnds = (4* ( dw %      75   ) +1 ) /3; /* convert from 1/75 to 1/100 */
  /* g_toc [i].bFlags contains two fields:
       bits 7-4 (ADR) : 0 no sub-q-channel information
                      : 1 sub-q-channel contains current position
		      : 2 sub-q-channel contains media catalog number
		      : 3 sub-q-channel contains International Standard
		                                 Recording Code ISRC
		      : other values reserved
       bits 3-0 (Control) :
       bit 3 : when set indicates there are 4 audio channels else 2 channels
       bit 2 : when set indicates this is a data track else an audio track
       bit 1 : when set indicates digital copy is permitted else prohibited
       bit 0 : when set indicates pre-emphasis is present else not present
  */

  if ( (global.verbose & SHOW_SUMMARY) != 0 ) {
    unsigned ii;

    /* summary */
    count_audio_trks = 0;
    i = 0;
    while ( i < cdtracks ) {
      int from;

      from = g_toc [i].bTrack;
      while ( i < cdtracks && g_toc [i].bFlags == g_toc [i+1].bFlags ) i++;
      if (i >= cdtracks) i--;
      
      if (g_toc[i].bFlags & 4) {
        fputs( " DATAtrack recorded      copy-permitted tracktype\n" , stderr);
      	fprintf(stderr, "     %2d-%2d %13.13s %14.14s      data\n",from,g_toc [i].bTrack,
			g_toc [i].bFlags & 1 ? "incremental" : "uninterrupted", /* how recorded */
			g_toc [i].bFlags & 2 ? "yes" : "no" /* copy-perm */
                       );
      } else { 
        fputs( "AUDIOtrack pre-emphasis  copy-permitted tracktype channels\n" , stderr);
	fprintf(stderr, "     %2d-%2d %12.12s  %14.14s     audio    %1c\n",from,g_toc [i].bTrack,
			g_toc [i].bFlags & 1 ? "yes" : "no", /* pre-emph */
			g_toc [i].bFlags & 2 ? "yes" : "no", /* copy-perm */
			g_toc [i].bFlags & 8 ? '4' : '2'
                       );
	count_audio_trks++;
      }
      i++;
    }
    fprintf ( stderr, 
	     "Table of Contents: total tracks:%u, (total time %u:%02u.%02u)\n",
	     cdtracks, mins, secnds, centi_secnds );

    for ( i = 0, ii = 0; i < cdtracks; i++ ) {
      if ( g_toc [i].bTrack <= MAXTRK ) {
	dw = (unsigned long) (g_toc[i+1].dwStartSector - g_toc[i].dwStartSector /* + 150 - 150 */);
	mins         =         dw / ( 60*75 );
	secnds       =       ( dw % ( 60*75 )) / 75;
	centi_secnds = ( 4 * ( dw %      75 ) + 1 ) / 3;
	if ( (g_toc [i].bFlags & CDROM_DATA_TRACK) != 0 ) {
		fprintf ( stderr, " %2u.[%2u:%02u.%02u]",
			g_toc [i].bTrack,mins,secnds,centi_secnds );
	} else {
		fprintf ( stderr, " %2u.(%2u:%02u.%02u)",
			g_toc [i].bTrack,mins,secnds,centi_secnds );
	}
        ii++;
      }
      if ( ii % 5 == 0 )
	fputs( "\n", stderr );
      else
	fputc ( ',', stderr );
    }
    if ( (ii+1) % 5 != 0 )
      fputs( "\n", stderr );

    if ((global.verbose & SHOW_STARTPOSITIONS) != 0) {
      fputs ("\nTable of Contents: starting sectors\n", stderr);
      for ( i = 0; i < cdtracks; i++ ) {
	fprintf ( stderr, " %2u.(%8u)", g_toc [i].bTrack, g_toc[i].dwStartSector
#ifdef DEBUG_CDDB
	+150
#endif
);
	if ( (i+1) % 5 == 0 )
	  fputs( "\n", stderr );
	else
	  fputc ( ',', stderr );
      }
      fprintf ( stderr, " lead-out(%8u)", g_toc[i].dwStartSector);
      fputs ("\n", stderr);
    }
    if (global.quiet == 0)
      fprintf(stderr, "CDDB discid: 0x%08lx\n", (unsigned long) global.cddb_id);
    if (have_CD_text != 0) {
      if (global.quiet == 0)
	fprintf(stderr, "CD-Text detected\n");
      dump_cdtext_info();
    }
    if (have_CD_extra != 0) {
      if (global.quiet == 0)
	fprintf(stderr, "CD-Extra (CD-Plus) detected\n");
      dump_extra_info();
    }
    if ((global.verbose & SHOW_TITLES) != 0) {
      if ( global.disctitle != NULL ) {
        fprintf( stderr, "Album title: '%s'\n", global.disctitle);
      }

      for ( i = 0; i < cdtracks; i++ ) {
	if ( global.tracktitle[i] != NULL ) {
	  fprintf( stderr, "Track %2u: '%s'\n", i+1, global.tracktitle[i]);
        }
      }
#if defined CDDB_SUPPORT
      if (count_audio_trks > 0 && global.no_cddbfile == 0) {
	emit_cddb_form(global.fname_base);
      }
#endif
    }
  }
}

void Read_MCN_ISRC()
{
  unsigned i;

  if ((global.verbose & SHOW_MCN) != 0) {
    subq_chnl *sub_ch;
    subq_catalog *subq_cat = NULL;

    /* get and display Media Catalog Number ( one per disc ) */
    fprintf(stderr, "scanning for MCN...");
    
    sub_ch = ReadSubQ( GET_CATALOGNUMBER,0);

#define EXPLICIT_READ_MCN_ISRC 1
#if EXPLICIT_READ_MCN_ISRC == 1 /* TOSHIBA HACK */
    if (Toshiba3401() != 0 && global.quiet == 0 && 
	(sub_ch != 0 || (((subq_catalog *)sub_ch->data)->mc_valid & 0x80))) {
      /* no valid MCN yet. do more searching */
      unsigned long h = g_toc[0].dwStartSector;
    
      while (h <= g_toc[0].dwStartSector + 100) {
	if (Toshiba3401())
	  ReadCdRom( RB_BASE->data, h, global.nsectors);
	sub_ch = ReadSubQ( GET_CATALOGNUMBER,0);
	if (sub_ch != NULL) {
	  subq_cat = (subq_catalog *) sub_ch->data;
	  if ((subq_cat->mc_valid & 0x80) != 0) {
	    break;
	  }
	}
	h += global.nsectors;
      }
    }
#endif

    if (sub_ch != NULL)
      subq_cat = (subq_catalog *)sub_ch->data;
  
    if (sub_ch != NULL && (subq_cat->mc_valid & 0x80) != 0 && global.quiet == 0) {

      /* unified format guesser:
       * format MCN all digits in bcd
       *     1                                  13
       * A: ab cd ef gh ij kl m0  0  0  0  0  0  0  Plextor 6x Rel. 1.02
       * B: 0a 0b 0c 0d 0e 0f 0g 0h 0i 0j 0k 0l 0m  Toshiba 3401
       * C: AS AS AS AS AS AS AS AS AS AS AS AS AS  ASCII SCSI-2 Plextor 4.5x and 6x Rel. 1.06
       */
      unsigned char *cp = subq_cat->media_catalog_number;
      if (!(cp[8] | cp[9] | cp[10] | cp[11] | cp[12]) &&
	  ((cp[0] & 0xf0) | (cp[1] & 0xf0) | (cp[2] & 0xf0) | 
	   (cp[3] & 0xf0) | (cp[4] & 0xf0) | (cp[5] & 0xf0) | 
	   (cp[6] & 0xf0))) {
	/* reformat A: to B: */
	cp[12] = cp[6] >> 4;
	cp[11] = cp[5] & 0xf;
	cp[10] = cp[5] >> 4;
	cp[ 9] = cp[4] & 0xf;
	cp[ 8] = cp[4] >> 4;
	cp[ 7] = cp[3] & 0xf;
	cp[ 6] = cp[3] >> 4;
	cp[ 5] = cp[2] & 0xf;
	cp[ 4] = cp[2] >> 4;
	cp[ 3] = cp[1] & 0xf;
	cp[ 2] = cp[1] >> 4;
	cp[ 1] = cp[0] & 0xf;
	cp[ 0] = cp[0] >> 4;
      }
      if (!isdigit(cp[0])) {
	if (memcmp(subq_cat->media_catalog_number,"\0\0\0\0\0\0\0\0\0\0\0\0\0",13) != 0)
	  sprintf((char *) subq_cat->media_catalog_number, 
		  "%1.1X%1.1X%1.1X%1.1X%1.1X%1.1X%1.1X%1.1X%1.1X%1.1X%1.1X%1.1X%1.1X", 
		  subq_cat->media_catalog_number [0],
		  subq_cat->media_catalog_number [1],
		  subq_cat->media_catalog_number [2],
		  subq_cat->media_catalog_number [3],
		  subq_cat->media_catalog_number [4],
		  subq_cat->media_catalog_number [5],
		  subq_cat->media_catalog_number [6],
		  subq_cat->media_catalog_number [7],
		  subq_cat->media_catalog_number [8],
		  subq_cat->media_catalog_number [9],
		  subq_cat->media_catalog_number [10],
		  subq_cat->media_catalog_number [11],
		  subq_cat->media_catalog_number [12]
		);
      }
      if (memcmp(subq_cat->media_catalog_number,"0000000000000",13) != 0) {
	memcpy((char *)MCN, (char *)subq_cat->media_catalog_number, 13);
        MCN[13] = 0;
	fprintf(stderr, "\rMedia catalog number: %13.13s\n", subq_cat->media_catalog_number);
      }
    }
  }

  if ((global.verbose & SHOW_ISRC) != 0) {
    subq_chnl *sub_ch;

    /* get and display Track International Standard Recording Codes 
       (for each track) */
    for ( i = 0; i < cdtracks; i++ ) {
      subq_track_isrc * subq_tr;
      unsigned j;

      fprintf(stderr, "\rscanning for ISRCs: %d ...", i+1);

      subq_tr = NULL;
      sub_ch = ReadSubQ( GET_TRACK_ISRC,i+1);

#if EXPLICIT_READ_MCN_ISRC == 1 /* TOSHIBA HACK */
      if (Toshiba3401() != 0) {
	j = (g_toc[i].dwStartSector/100 + 1) * 100;
	do {
	  ReadCdRom(  RB_BASE->data, j, global.nsectors);
	  sub_ch = ReadSubQ( GET_TRACK_ISRC, g_toc[i].bTrack);
	  if (sub_ch != NULL) {
	    subq_tr = (subq_track_isrc *) sub_ch->data;
	    if (subq_tr != NULL && (subq_tr->tc_valid & 0x80) != 0)
	      break;
	  }
	  j += global.nsectors;
	} while (j < (g_toc[i].dwStartSector/100 + 1) * 100 + 100);
      }
#endif
    
      if (sub_ch != NULL)
	subq_tr = (subq_track_isrc *)sub_ch->data;

      if (sub_ch != NULL && (subq_tr->tc_valid & 0x80) && global.quiet == 0) {
	unsigned char p_start[16];
	unsigned char *p = p_start;
	unsigned char *cp = subq_tr->track_isrc;

#if 0
	int ijk;
	for (ijk = 0; ijk < 15; ijk++) {
		fprintf(stderr, "%02x  ", cp[ijk]);
        }
	fputs("", stderr);
#endif
	/* unified format guesser:
	 * there are 60 bits and 15 bytes available.
	 * 5 * 6bit-items + two zero fill bits + 7 * 4bit-items
	 *
	 * A: ab cd ef gh ij kl mn o0 0  0  0  0  0  0  0  Plextor 6x Rel. 1.02
	 * B: 0a 0b 0c 0d 0e 0f 0g 0h 0i 0j 0k 0l 0m 0n 0o Toshiba 3401
	 * C: AS AS AS AS AS AS AS AS AS AS AS AS AS AS AS ASCII SCSI-2
	 * eg 'G''B'' ''A''0''7'' ''6''8'' ''0''0''2''7''0' makes most sense
	 * D: 'G''B''A''0''7''6''8''0''0''2''7''0'0  0  0  Plextor 6x Rel. 1.06 and 4.5x R. 1.01 and 1.04
	 */
	if (!(cp[8] | cp[9] | cp[10] | cp[11] | cp[12] | cp[13] | cp[14]) &&
	    ((cp[0] & 0xf0) | (cp[1] & 0xf0) | (cp[2] & 0xf0) | 
	     (cp[3] & 0xf0) | (cp[4] & 0xf0) | (cp[5] & 0xf0) | 
	     (cp[6] & 0xf0) | (cp[7] & 0xf0))) {
	  /* reformat A: to B: */
	  cp[14] = cp[7] >> 4;
	  cp[13] = cp[6] & 0xf;
	  cp[12] = cp[6] >> 4;
	  cp[11] = cp[5] & 0xf;
	  cp[10] = cp[5] >> 4;
	  cp[ 9] = cp[4] & 0xf;
	  cp[ 8] = cp[4] >> 4;
	  cp[ 7] = cp[3] & 0xf;
	  cp[ 6] = cp[3] >> 4;
	  cp[ 5] = cp[2] & 0xf;
	  cp[ 4] = cp[2] >> 4;
	  cp[ 3] = cp[1] & 0xf;
	  cp[ 2] = cp[1] >> 4;
	  cp[ 1] = cp[0] & 0xf;
	  cp[ 0] = cp[0] >> 4;
	}
      
	/* If not yet in ASCII format, do the conversion */
	if (!isupper(cp[0]) || !isupper(cp[1])) {
	  /* coding table for International Standard Recording Code */
	  static char bin2ISRC[] = 
	    {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
	       0,0,0,0,0,0,'@',
	       'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K',
	       'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V',
	       'W', 'X', 'Y', 'Z'
	   };
	
	  /* build 6-bit vector of coded values */
	  unsigned ind;
	  int bits;
	
	  ind = (cp[0] << 26) +
	        (cp[1] << 22) +
	        (cp[2] << 18) + 
	        (cp[3] << 14) +
	        (cp[4] << 10) +
	        (cp[5] << 6) +
	        (cp[6] << 2) +
	        (cp[7] >> 2);
	  
	  /* decode ISRC due to IEC 908 */
	  for (bits = 0; bits < 30; bits +=6) {
	    int binval = (ind & ((unsigned long) 0x3fL << (24L-bits))) >> (24L-bits);
	    *p++ = bin2ISRC[binval];
	  
	    /* insert a space after two country characters for legibility */
	    if (bits == 6)
	      *p++ = ' ';
	  }
	  
	  /* format year and serial number */
	  sprintf ((char *)p, " %.1X%.1X %.1X%.1X%.1X%.1X%.1X",
		   subq_tr->track_isrc [8],
		   subq_tr->track_isrc [9],
		   subq_tr->track_isrc [10],
		   subq_tr->track_isrc [11],
		   subq_tr->track_isrc [12],
		   subq_tr->track_isrc [13],
		   subq_tr->track_isrc [14]
		   ); 
	} else {
	  /* It is really in ASCII, surprise */
	  int ii;
	  for (ii = 0; ii < 12; ii++) {
	    if ((ii == 2 || ii == 5 || ii == 7) && cp[ii] != ' ')
	      *p++ = ' ';
	    *p++ = cp[ii];
	  }
	  if (p - p_start >= 16)
	    *(p_start + 15) = '\0';
	  else
	    *p = '\0';
	}
	memcpy((char *)g_toc[i].ISRC, (const char *)p_start, 16);
	fprintf (stderr, "\rT: %2d ISRC: %s\n", i+1, g_toc[i].ISRC);
	fflush(stderr); 
      }
    }
  fputs("\n", stderr);
  }
}



static int GetIndexOfSector( sec )
	unsigned sec;
{
    subq_chnl *sub_ch;

    if (-1 == Play_at( sec, 1)) return -1;

    sub_ch = ReadSubQ( GET_POSITIONDATA,0);

    return sub_ch ? sub_ch->index == 244 ? 1 : sub_ch->index : -1;
}


static int binary_search(searchInd, Start, End)
	int searchInd;
	unsigned Start;
	unsigned End;
{
      int l = Start; int r = End; int x = 0;
      int ind;
      while ( r >= l ) {
	  x = ( l + r ) / 2;
	  /* try to avoid seeking */
	  ind = GetIndexOfSector(x);
	  if ( searchInd == ind ) {
	      break;
	  } else
	      if ( searchInd < ind ) r = x - 1;
	      else	     	     l = x + 1;
      }
      if ( r >= l ) {
        /* Index found. Now find the first position of this index */
	/* l=LastPos	x=found		r=NextPos */
        r = x;
	while ( l < r-1 ) {
	  x = ( l + r ) / 2;
	  /* try to avoid seeking */
	  ind = GetIndexOfSector(x);
	  if ( searchInd == ind ) {
	      r = x;
	  } else {
	      l = x;
	  }
        }
	return r;
      }

      return -1;
}

#undef DEBUG_INDLIST
/* experimental code */
/* search for indices (audio mode required) */
unsigned ScanIndices( track, cd_index )
	unsigned track;
	unsigned cd_index;
{
  /* scan for indices. */
  /* look at last sector of track. */
  /* when the index is not equal 1 scan by bipartition 
   * for offsets of all indices */

  unsigned i; 
  unsigned starttrack, endtrack;
  unsigned startindex, endindex;

  unsigned j;
  int LastIndex=0; 
  unsigned StartSector;
  unsigned retval = 0;

  index_list *baseindex_pool;
  index_list *indexentry;
  index_list *last_index_entry;

  if (!global.quiet && !(global.verbose & SHOW_INDICES))
    fprintf(stderr, "seeking index start ...");
  if (cd_index != 1) {
    starttrack = track; endtrack = track;
  } else {
    starttrack = 1; endtrack = cdtracks;
  }
  baseindex_pool = (index_list *) malloc( sizeof(index_list) * (endtrack - starttrack + 1));
#ifdef DEBUG_INDLIST
  fprintf(stderr, "index0-mem-pool %p\n", baseindex_pool);
#endif

  for (i = starttrack; i <= endtrack; i++) {
    if ( global.verbose & SHOW_INDICES ) { 
	fprintf( stderr, "index scan: %d...\r", i ); 
	fflush (stderr);
    }
    if ( g_toc [i-1].bFlags & CDROM_DATA_TRACK )
	continue;/* skip nonaudio tracks */
    StartSector = GetStartSector(i);
    LastIndex = GetIndexOfSector(GetEndSector(i));
    if (LastIndex == 0) {
      LastIndex = GetIndexOfSector(GetEndSector(i)-2*75);
    }

    /* register first index entry for this track */
    if (baseindex_pool != NULL) {
#ifdef DEBUG_INDLIST
  fprintf(stderr, "audio track %d, arrindex %d\n", i, i - starttrack);
#endif
      baseindex_pool[i - starttrack].next = NULL;
      baseindex_pool[i - starttrack].frameoffset = 0;
      global.trackindexlist[i-1] = &baseindex_pool[i - starttrack];
#ifdef DEBUG_INDLIST
  fprintf(stderr, "indexbasep %p\n", &baseindex_pool[i - starttrack]);
#endif
    } else {
      global.trackindexlist[i-1] = NULL;
    }
    if (LastIndex < 2) {
      continue;
    }
    last_index_entry = global.trackindexlist[i-1];

    if ((global.verbose & SHOW_INDICES) && LastIndex > 1)
	fprintf(stderr, "track %2d has %d indices, index table (pairs of index, frame offset)\n", i, LastIndex);

    if (cd_index != 1) {
	startindex = cd_index; endindex = cd_index;
    } else {
	startindex = 2; endindex = LastIndex;
    }
    for (j = startindex; j <= endindex; j++) {
      int IndexOffset;

      /* this track has indices */

      /* do a binary search */
      IndexOffset = binary_search(j, StartSector, GetEndSector(i));

      /* register higher index entries */
      if (last_index_entry != NULL) {
        indexentry = (index_list *) malloc( sizeof(index_list) );
      } else {
        indexentry = NULL;
      }
      if (indexentry != NULL) {
        indexentry->next = NULL;
        last_index_entry->next = indexentry;
        last_index_entry = indexentry;
      } else {
#if defined INFOFILES
        fprintf( stderr, "No memory for index lists. Index positions\nwill not be written in info file!\n");
#endif
      }

      if ( IndexOffset == -1 ) {
        if (indexentry != NULL) {
          indexentry->frameoffset = -1;
        }
	fprintf(stderr, " %2u N/A    ",j);
      } else {
	  if (global.verbose & SHOW_INDICES)
	    fprintf(stderr, 
#if 0
		    "Ind. %2u (%6lu),   ",
#else
		    "%2u,%6lu ",
#endif
		    j,
		    IndexOffset-GetStartSector(i));
          if (indexentry != NULL) {
            indexentry->frameoffset = IndexOffset-GetStartSector(i);
          }
	  StartSector = IndexOffset;
	  if (track == i && cd_index == j) {
	     retval = IndexOffset-GetStartSector(i);
	  }
      }
    }
    fputs("\n", stderr);
  }
  return retval;
}
